/*
 * Phone Phreak - Brett Walach - 2022
 */
#include <vectrex.h>
#include "controller.h"
#include "sound.h"

const struct sound_adsr_table_t adsr_table_blue =
{
    .amplitude = {
        (int)0xff,(int)0xff,(int)0xff,(int)0xff,
        (int)0xff,(int)0xff,(int)0xff,(int)0xff,
        (int)0xff,(int)0xff,(int)0xff,(int)0xff,
        (int)0xff,(int)0xff,(int)0xff,(int)0xff,
    }
};

const struct sound_adsr_table_t adsr_table_red =
{
    .amplitude = {
        (int)0x00,(int)0xff,(int)0xff,(int)0xff,
        (int)0x00,(int)0x00,(int)0x00,(int)0x00,
        (int)0x00,(int)0x00,(int)0x00,(int)0x00,
        (int)0x00,(int)0x00,(int)0x00,(int)0x00
    }
};

const struct sound_twang_table_t twang_table =
{
    .frequency = {
        (int)0x00,(int)0x00,(int)0x00,(int)0x00,
        (int)0x00,(int)0x00,(int)0x00,(int)0x00
    }
};

const struct sound_music_t blue_box =
{
    .adsr_table = (struct sound_adsr_table_t*) &adsr_table_blue,
    .twang_table = (struct sound_twang_table_t*) &twang_table,
    {
        __N_E7,10, // E7 = 2637.02 Hz (close enough to 2600 Hz ???)
        __N_E7,10, //  |
        __N_E7,10, //  |
        __N_E7,10, //  |
        __N_E7,10, //  |
        __N_E7,10, //  |
        __N_E7,10, //  |
        __N_E7,10, //  |
        __N_E7,10, //  |
        __N_E7,10, //  |
        __N_E7,10, //  |
        __N_E7,10, //  | add more or less to control duration
        0, 128 // 0x80 is end marker for music, frequency is not played so 0
    }
};

const struct sound_music_t red_box_nickel =
{
    .adsr_table = (struct sound_adsr_table_t*) &adsr_table_red,
    .twang_table = (struct sound_twang_table_t*) &twang_table,
    {
        128 + __N_A6, 128 + __N_C7, __N_A6, 3, // A6 = 1760 Hz + C7 2093 Hz (close enough to 1700 + 2200 Hz ???)
        0, 3,
        0, 255 // 0x80 is end marker for music, frequency is not played so 0
    }
};

const struct sound_music_t red_box_dime =
{
    .adsr_table = (struct sound_adsr_table_t*) &adsr_table_red,
    .twang_table = (struct sound_twang_table_t*) &twang_table,
    {
        128 + __N_A6, 128 + __N_C7, __N_A6, 3, // A6 = 1760 Hz + C7 2093 Hz (close enough to 1700 + 2200 Hz ???)
        0, 3,
        128 + __N_A6, 128 + __N_C7, __N_A6, 3, // A6 = 1760 Hz + C7 2093 Hz (close enough to 1700 + 2200 Hz ???)
        0, 3,
        0, 255 // 0x80 is end marker for music, frequency is not played so 0
    }
};

const struct sound_music_t red_box_quarter =
{
    // .adsr_table = (struct sound_adsr_table_t*) &Vec_ADSR_FADE1,
    .adsr_table = (struct sound_adsr_table_t*) &adsr_table_red,
    .twang_table = (struct sound_twang_table_t*) &twang_table,
    {
        128 + __N_A6, 128 + __N_C7, __N_A6, 2, // A6 = 1760 Hz + C7 2093 Hz (close enough to 1700 + 2200 Hz ???)
        0, 2,
        128 + __N_A6, 128 + __N_C7, __N_A6, 2, // A6 = 1760 Hz + C7 2093 Hz (close enough to 1700 + 2200 Hz ???)
        0, 2,
        128 + __N_A6, 128 + __N_C7, __N_A6, 2, // A6 = 1760 Hz + C7 2093 Hz (close enough to 1700 + 2200 Hz ???)
        0, 2,
        128 + __N_A6, 128 + __N_C7, __N_A6, 2, // A6 = 1760 Hz + C7 2093 Hz (close enough to 1700 + 2200 Hz ???)
        0, 2,
        128 + __N_A6, 128 + __N_C7, __N_A6, 2, // A6 = 1760 Hz + C7 2093 Hz (close enough to 1700 + 2200 Hz ???)
        0, 2,
        0, 255 // 0x80 is end marker for music, frequency is not played so 0
    }
};

/*
 * Some defines, Maximal brightness is $7f, highest not set!
 * Max Scale would be $ff, well here we take only $f0!
 */
#define MAX_BRIGHTNESS (0x7f)
#define MAX_SCALE (0xf0)

/*
 * For variable variables ALWAYS leave them uninitialized, this way
 * the compiler puts them into the BSS ram section in vectrex ram
 * area from c880 onwards.
 *
 * All non BSS memory should be declared constant!
 * (This is still leaves the option of auto variables in a
 * functiion, which will takes it needed men from the stack)
 *
 */
void* current_song;

/*
 * If you declare a function 'static inline' it is pretty much sure that
 * the compiler will inline it completely, the function will not
 * appear in any way in the module.
 *
 * If you leave out the static, the compiler assumes it might be used
 * globally, and thus leaves the function in the code, even if
 * it is inlined everywhere!
 */
/*
 * This Funktion handles all startup code, needed for vectrex to work
 * correctly, recallibrating the beam and making sure that some sound
 * is played (if wanted).
 */
static inline void start_one_vectrex_round(void)
{
  DP_to_C8();                        /* vectrex internal... dp must point */
  Init_Music_chk(current_song);    /* to c800, could make a function which */
  Wait_Recal();                       /* sets this up allright... */
  Do_Sound();
}

/*
 * This function sets up a piece of music to be played from the start
 * of the next round on onward...
 */
static inline void play_song(void* song)
{
  Vec_Music_Flag = 1;       /* A makro to write to a specific memory */
  current_song = song;                /* address */
}

/*
 * Our main function we start of here...
 * we should make sure that we never return from here, or vectrex will
 * be surely bothered!
 */
int main(void)
{
  unsigned char last_button = 0;        /* one auto variable... remember the */
  check_buttons();                       /* last pressed button */
  while (1)                             /* never to return... */
  {
    start_one_vectrex_round();          /* start 'de round */
    VIA_t1_cnt_lo = MAX_SCALE;               /* set scale factor */
    Intensity_a(MAX_BRIGHTNESS);          /* set some brightness */
    Print_Str_d(110,-70, "PICK A BOX\x80");        /* print on screen */
    Print_Str_d(70,-70, "1 - BLUE 2600\x80");
    Print_Str_d(30,-70, "2 - RED NICKEL\x80");
    Print_Str_d(-10,-70, "3 - RED DIME\x80");
    Print_Str_d(-50,-70,"4 - RED QUARTER\x80");
    if ((button_1_1_pressed()) && (last_button != 1)) /* check the button 1 */
    {
      play_song((void*)&blue_box);                 /* play a song */
      last_button = 1;                             /* and remember this button */
    } else
    if ((button_1_2_pressed()) && (last_button != 2)) /* check the button 2 */
    {
      play_song((void*)&red_box_nickel);           /* play a song */
      last_button = 2;                             /* and remember this button */
    } else
    if ((button_1_3_pressed()) && (last_button != 3)) /* check the button 3 */
    {
      play_song((void*)&red_box_dime	);           /* play a song */
      last_button = 3;                             /* and remember this button */
    } else
    if ((button_1_4_pressed()) && (last_button != 4)) /* check the button 4 */
    {
      play_song((void*)&red_box_quarter	);         /* play a song */
      last_button = 4;                             /* and remember this button */
    }
    check_buttons();                               /* read again for next round */
  } /* while (true) */
}

/* END OF FILE */
